/**
 * ReportCard is an aggregate of GradeRecord objects.
 * This version uses an array of GradeRecords
 * 
 * @author JP Vergara
 * @version 1.0
 */
public class ReportCard
{
    private GradeRecord[] grades;
    private int count;  // represents the actual number of GradeRecord objects
    public static final int MAX = 10; // the maximum number of GradeRecord objects

    /**
     * Constructor for objects of class ReportCard
     */
    public ReportCard()
    {
        grades = new GradeRecord[MAX];
        count = 0;
    }
    
    /*
     * Adds a grade record with the specified subject name
     * and grade
     * 
     * @param subject  the subject name; e.g., "English"
     * @param grade    the grade for that subject; e.g., 85.5
     */
    public void addGradeRecord( String subject, double grade )
    {
        GradeRecord g = new GradeRecord( subject, grade );
        grades[count] = g;  // conveniently, count points to the next available slot
        count++; // increment because now we have 1 more record than before
    }

    /*
     * Updates the grade record associated with the specified
     * subject name with a new grade;
     * if no such grade record exists, do nothing
     * 
     * @param subject  the subject name of the record
     * @param grade    the new grade for that subject
     */
    public void updateGradeRecord( String subject, double grade )
    {
        GradeRecord g = findRecord( subject );
        if ( g != null )
        {
           g.changeGrade( grade );
        }
    }
    
    /*
     * Returns the grade associated with the specified
     * subject name
     * if no such grade record exists, return -1
     * 
     * @param subject  the subject name of the record
     * @return the grade for that subject
     */
    public double getGrade( String subject )
    {
        GradeRecord g = findRecord( subject );
        if ( g == null )
        {
            return -1;
        }
        else
        {
           return g.getGrade();
        }
    }
    
    /*
     * Returns the letter grade associated with the specified
     * subject name
     * if no such grade record exists, return "*"
     * 
     * @param subject  the subject name of the record
     * @return the letter grade for that subject
     */
    public String getLetterGrade( String subject )
    {
        GradeRecord g = findRecord( subject );
        if ( g == null )
        {
            return "*";
        }
        else
        {
           return g.getLetterGrade();
        }
    }
    
    /*
     * Returns the average of all grades;
     * there are 0 grade records, return 0;
     * 
     * @return the average
     */
    public double getAverage()
    {
        if ( count == 0 )
           return 0;
        double sum = 0;
        for( int i = 0; i < count; i++ )
        {
            sum = sum + grades[i].getGrade();
        }
        return sum/count;
    }
    
    /*
     * Returns the grade record associated 
     * with the specified subject name;
     * returns null if no such record exists.
     * Method is private because it is used only internally
     * 
     * @param subject  the subject name of the record
     * @return the grade record
     */
    private GradeRecord findRecord( String subject )
    {
        for( int i = 0; i < count; i++ )
        {
            if ( grades[i].getSubject().equals( subject ) )
            {
                return grades[i];
            }
        }
        return null; // record not found
    }

}
